﻿//-------------------------------------------------------------------------------------
// SID Monitor - Utility for Sudden Ionospheric Disturbances Monitoring Stations
// Copyright (C) 2006 - Lionel Loudet
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//-------------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace SID_monitor
{
    public partial class PanelGraph : UserControl
    {

        public PanelGraph()
        {
            InitializeComponent();

            this.numericUpDownRefreshUpdate.Value = Math.Round((decimal)(SID_monitor.Properties.Settings.Default.GraphUpdate / 1000 / 60));

            this.colorButtonCivilTw.CenterColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.CivilTwilightColor);
            this.colorButtonNautiTw.CenterColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.NauticalTwilightColor);
            this.colorButtonAstroTw.CenterColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.AstronomicalTwilightColor);
            this.colorButtonNight.CenterColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.NightColor);

            this.colorButtonFailures.CenterColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.FailuresColor);

            this.colorButtonColorExample.CenterColor = Color.Lime; // Lime is a good starting value for illustrating the shading effect
            this.labelExample0.BackColor = Color.Lime;

            if (SID_monitor.Properties.Settings.Default.ShadeValue1 < 0)
            {
                this.numericUpDownShadeValue1.Value = 0.0M;
            }
            else if (SID_monitor.Properties.Settings.Default.ShadeValue1 > 100)
            {
                this.numericUpDownShadeValue1.Value = 1.0M;
            }
            else
            {
                this.numericUpDownShadeValue1.Value = SID_monitor.Properties.Settings.Default.ShadeValue1 / 100.0M;
            }

            if (SID_monitor.Properties.Settings.Default.ShadeValue2 < 0)
            {
                this.numericUpDownShadeValue2.Value = 0.0M;
            }
            else if (SID_monitor.Properties.Settings.Default.ShadeValue2 > 100)
            {
                this.numericUpDownShadeValue2.Value = 1.0M;
            }
            else
            {
                this.numericUpDownShadeValue2.Value = SID_monitor.Properties.Settings.Default.ShadeValue2 / 100.0M;
            }

            if (SID_monitor.Properties.Settings.Default.ShadeValue3 < 0)
            {
                this.numericUpDownShadeValue3.Value = 0.0M;
            }
            else if (SID_monitor.Properties.Settings.Default.ShadeValue3 > 100)
            {
                this.numericUpDownShadeValue3.Value = 1.0M;
            }
            else
            {
                this.numericUpDownShadeValue3.Value = SID_monitor.Properties.Settings.Default.ShadeValue3 / 100.0M;
            }

        }

        void ColorButton_Click(object sender, System.EventArgs e)
        {
            ColorButton callingButton = (ColorButton)sender;
            Point p = new Point(callingButton.Left, callingButton.Top + callingButton.Height);
            p = PointToScreen(p);

            ColorPaletteDialog clDlg = new ColorPaletteDialog(p.X, p.Y);

            clDlg.ShowDialog();

            if (clDlg.DialogResult == DialogResult.OK)
            {
                callingButton.CenterColor = clDlg.Color;

                if (callingButton.Equals(this.colorButtonColorExample))  // we add this to synchronise the example label area background color to the selected color
                {
                    this.labelExample0.BackColor = clDlg.Color;
                }
            }

            Invalidate();

            clDlg.Dispose();
        }

        #region Threshold numeric up-down and trackbar synchronisation
        /// <summary>
        /// We synchronise the trackBar.
        /// We must keep: ShadeValue 1 <= ShadeValue2 <= ShadeValue3
        /// </summary>
        private void numericUpDownShadeValue1_ValueChanged(object sender, EventArgs e)
        {
            if (this.trackBarShadeValue1.Value != (int)(this.numericUpDownShadeValue1.Value * 100))
            {
                this.trackBarShadeValue1.Value = (int)(this.numericUpDownShadeValue1.Value * 100);
            }

            if (this.numericUpDownShadeValue1.Value > this.numericUpDownShadeValue2.Value)
            {
                this.numericUpDownShadeValue2.Value = this.numericUpDownShadeValue1.Value;
            }

            this.labelExample1.BackColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.LighterColor(this.labelExample0.BackColor.ToArgb(), this.trackBarShadeValue1.Value / 100.0));
        }

        private void numericUpDownShadeValue2_ValueChanged(object sender, EventArgs e)
        {
            if (this.trackBarShadeValue2.Value != (int)(this.numericUpDownShadeValue2.Value * 100))
            {
                this.trackBarShadeValue2.Value = (int)(this.numericUpDownShadeValue2.Value * 100);
            }

            if (this.numericUpDownShadeValue2.Value > this.numericUpDownShadeValue3.Value)
            {
                this.numericUpDownShadeValue3.Value = this.numericUpDownShadeValue2.Value;
            }

            if (this.numericUpDownShadeValue2.Value < this.numericUpDownShadeValue1.Value)
            {
                this.numericUpDownShadeValue1.Value = this.numericUpDownShadeValue2.Value;
            }

            this.labelExample2.BackColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.LighterColor(this.labelExample0.BackColor.ToArgb(), this.trackBarShadeValue2.Value / 100.0));
        }

        private void numericUpDownShadeValue3_ValueChanged(object sender, EventArgs e)
        {
            if (this.trackBarShadeValue3.Value != (int)(this.numericUpDownShadeValue3.Value * 100))
            {
                this.trackBarShadeValue3.Value = (int)(this.numericUpDownShadeValue3.Value * 100);
            }

            if (this.numericUpDownShadeValue3.Value < this.numericUpDownShadeValue2.Value)
            {
                this.numericUpDownShadeValue2.Value = this.numericUpDownShadeValue3.Value;
            }

            this.labelExample3.BackColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.LighterColor(this.labelExample0.BackColor.ToArgb(), this.trackBarShadeValue3.Value / 100.0));
        }

        /// <summary>
        /// We synchronise the numericUpDown.
        /// </summary>
        private void trackBarShadeValue1_Scroll(object sender, EventArgs e)
        {
            if (this.numericUpDownShadeValue1.Value != (decimal)(this.trackBarShadeValue1.Value / 100.0))
            {
                this.numericUpDownShadeValue1.Value = (decimal)(this.trackBarShadeValue1.Value / 100.0);
            }
        }

        private void trackBarShadeValue2_Scroll(object sender, EventArgs e)
        {
            if (this.numericUpDownShadeValue2.Value != (decimal)(this.trackBarShadeValue2.Value / 100.0))
            {
                this.numericUpDownShadeValue2.Value = (decimal)(this.trackBarShadeValue2.Value / 100.0);
            }
        }

        private void trackBarShadeValue3_Scroll(object sender, EventArgs e)
        {
            if (this.numericUpDownShadeValue3.Value != (decimal)(this.trackBarShadeValue3.Value / 100.0))
            {
                this.numericUpDownShadeValue3.Value = (decimal)(this.trackBarShadeValue3.Value / 100.0);
            }

        }
        #endregion

        #region Properties
        public int CivilTwilightColor
        {
            get
            {
                return this.colorButtonCivilTw.CenterColor.ToArgb();
            }
        }

        public int NautiTwilightColor
        {
            get
            {
                return this.colorButtonNautiTw.CenterColor.ToArgb();
            }
        }

        public int AstroTwilightColor
        {
            get
            {
                return this.colorButtonAstroTw.CenterColor.ToArgb();
            }
        }

        public int NightTwilightColor
        {
            get
            {
                return this.colorButtonNight.CenterColor.ToArgb();
            }
        }


        public int GraphUpdateValue
        {
            get
            {
                return (int)(this.numericUpDownRefreshUpdate.Value * 1000 * 60);
            }
        }

        public int FailuresColor
        {
            get
            {
                return this.colorButtonFailures.CenterColor.ToArgb();
            }
        }

        public int ShadeValue1
        {
            get
            {
                return (int)this.trackBarShadeValue1.Value;
            }
        }

        public int ShadeValue2
        {
            get
            {
                return (int)this.trackBarShadeValue2.Value;
            }
        }

        public int ShadeValue3
        {
            get
            {
                return (int)this.trackBarShadeValue3.Value;
            }
        }
        #endregion

        #region ValidData Event
        public event ValidDataEventHandler ValidData;

        protected virtual void OnValidData(validDataEventArgs e)
        {
            if (ValidData != null)
            {
                // Invokes the delegates. 
                ValidData(this, e);
            }
        }
        #endregion

        private void labelExample0_BackColorChanged(object sender, EventArgs e)
        {
            this.labelExample1.BackColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.LighterColor(this.labelExample0.BackColor.ToArgb(), this.trackBarShadeValue1.Value / 100.0));
            this.labelExample2.BackColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.LighterColor(this.labelExample0.BackColor.ToArgb(), this.trackBarShadeValue2.Value / 100.0));
            this.labelExample3.BackColor = Color.FromArgb(SID_monitor.Properties.Settings.Default.LighterColor(this.labelExample0.BackColor.ToArgb(), this.trackBarShadeValue3.Value / 100.0));
        }

       


    }
}
